#ifndef GLSL_PHOTOSHOP
#define GLSL_PHOTOSHOP

/*
photoshop.glsl
Implements photoshop style filters
*/

//Include the color ops and blend ops
#pragma lz append("lzsystem/shaders/postprocess/color.glsl")
#pragma lz append("lzsystem/shaders/postprocess/blendops.glsl")

//Given an HSL value a photoshop style hue/saturation/lightness filter is applied
vec3 HueSatHSL(vec3 hsl, float hue, float sat, float light) {
	//The filter
	hsl.x += hue;
	hsl.x -= floor(hsl.x / 360.0) * 360.0; //Cap at 0..360
	hsl.y = max(0, min(1, hsl.y + hsl.y * sat));
	hsl.z = max(0, min(1, hsl.z + light));

	return hsl;
}

//Given an RGB value performs a hue/saturation/lightness operation and returns an rgb modified color
vec3 HueSatRGB(vec3 rgb, float hue, float sat, float light) {
	//Convert to HSL, filter, convert back to RGB and return
	return HSL2RGB(HueSatHSL(RGB2HSL(rgb), hue, sat, light));
}

//Applies a photoshop style Hue filter to an HCY, HCV or HSL value and returns an HSL modified value
vec3 HueFilter(vec3 base, vec3 top) {
	return vec3(top.x, base.yz);
}

//Applies a photoshop style Saturation filter to an HCY, HCV or HSL value and returns an HSL modified value
vec3 SaturationFilter(vec3 base, vec3 top) {
	return vec3(base.x, top.y, base.z);
}

//Applies a photoshop style Color filter to an HCY, HCV or HSL value and returns an HSL modified value
vec3 ColorFilter(vec3 base, vec3 top) {
	return vec3(top.xy, base.z);
}

//Applies a photoshop style Luminosity filter to an HSL value and returns an HSL modified value
vec3 LuminosityFilter(vec3 base, vec3 top) {
	return vec3(base.xy, top.z);
}

//The following are similar filters to the four above but with implicit conversion to a color space
//Applies a photoshop style Hue filter to an RGB value converting it to HSL and returns an RGB modified value
vec3 HueHSL(vec3 base, vec3 top) {
	return HSL2RGB(HueFilter(RGB2HSL(base), RGB2HSL(top)));
}

//Applies a photoshop style Saturation filter to an RGB value converting it to HSL and returns an RGB modified value
vec3 SaturationHSL(vec3 base, vec3 top) {
	return HSL2RGB(SaturationFilter(RGB2HSL(base), RGB2HSL(top)));
}

//Applies a photoshop style Color filter to an RGB value converting it to HSL and returns an RGB modified value
vec3 ColorHSL(vec3 base, vec3 top) {
	return HSL2RGB(ColorFilter(RGB2HSL(base), RGB2HSL(top)));
}

//Applies a photoshop style Color filter to an RGB value converting it to HSL and returns an RGB modified value
vec3 LuminosityHSL(vec3 base, vec3 top) {
	return HSL2RGB(LuminosityFilter(RGB2HSL(base), RGB2HSL(top)));
}


//Applies a photoshop style Hue filter to an RGB value converting it to HCY and returns an RGB modified value
vec3 HueHCY(vec3 base, vec3 top) {
	return HCY2RGB(HueFilter(RGB2HCY(base), RGB2HCY(top)));
}

//Applies a photoshop style Saturation filter to an RGB value converting it to HCY and returns an RGB modified value
vec3 SaturationHCY(vec3 base, vec3 top) {
	return HCY2RGB(SaturationFilter(RGB2HCY(base), RGB2HCY(top)));
}

//Applies a photoshop style Color filter to an RGB value converting it to HCY and returns an RGB modified value
vec3 ColorHCY(vec3 base, vec3 top) {
	return HCY2RGB(ColorFilter(RGB2HCY(base), RGB2HCY(top)));
}

//Applies a photoshop style Color filter to an RGB value converting it to HCY and returns an RGB modified value
vec3 LuminosityHCY(vec3 base, vec3 top) {
	return HCY2RGB(LuminosityFilter(RGB2HCY(base), RGB2HCY(top)));
}


//Applies a photoshop style Hue filter to an RGB value converting it to HSY and returns an RGB modified value
vec3 HueHSY(vec3 base, vec3 top) {
	return HSY2RGB(HueFilter(RGB2HSY(base), RGB2HSY(top)));
}

//Applies a photoshop style Saturation filter to an RGB value converting it to HSY and returns an RGB modified value
vec3 SaturationHSY(vec3 base, vec3 top) {
	return HSY2RGB(SaturationFilter(RGB2HSY(base), RGB2HSY(top)));
}

//Applies a photoshop style Color filter to an RGB value converting it to HSY and returns an RGB modified value
vec3 ColorHSY(vec3 base, vec3 top) {
	return HSY2RGB(ColorFilter(RGB2HSY(base), RGB2HSY(top)));
}

//Applies a photoshop style Color filter to an RGB value converting it to HSY and returns an RGB modified value
vec3 LuminosityHSY(vec3 base, vec3 top) {
	return HSY2RGB(LuminosityFilter(RGB2HSY(base), RGB2HSY(top)));
}


//Applies a photoshop style Hue filter to an RGB value converting it to HSV and returns an RGB modified value
vec3 HueHSV(vec3 base, vec3 top) {
	return HSV2RGB(HueFilter(RGB2HSV(base), RGB2HSV(top)));
}

//Applies a photoshop style Saturation filter to an RGB value converting it to HSV and returns an RGB modified value
vec3 SaturationHSV(vec3 base, vec3 top) {
	return HSV2RGB(SaturationFilter(RGB2HSV(base), RGB2HSV(top)));
}

//Applies a photoshop style Color filter to an RGB value converting it to HSV and returns an RGB modified value
vec3 ColorHSV(vec3 base, vec3 top) {
	return HSV2RGB(ColorFilter(RGB2HSV(base), RGB2HSV(top)));
}

//Applies a photoshop style Color filter to an RGB value converting it to HSV and returns an RGB modified value
vec3 LuminosityHSV(vec3 base, vec3 top) {
	return HSV2RGB(LuminosityFilter(RGB2HSV(base), RGB2HSV(top)));
}

/*
photoshop.glsl end
*/

#endif